<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * Course model
 *
 * @property integer $id
 * @property string $name
 * @property string $description
 * @property integer $duration_hours
 * @property float $price
 * @property boolean $is_active
 * @property string $created_at
 * @property string $updated_at
 */
class Course extends ActiveRecord
{
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'courses';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['name'], 'required'],
            [['description'], 'string'],
            [['duration_hours'], 'integer', 'min' => 1],
            [['price'], 'number', 'min' => 0],
            [['is_active'], 'boolean'],
            [['name'], 'string', 'max' => 200],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'name' => 'Название курса',
            'description' => 'Описание',
            'duration_hours' => 'Продолжительность (часов)',
            'price' => 'Стоимость (руб.)',
            'is_active' => 'Активен',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * Получить список активных курсов
     */
    public static function getActiveCourses()
    {
        return self::find()
            ->where(['is_active' => true])
            ->orderBy(['name' => SORT_ASC])
            ->all();
    }

    /**
     * Получить список курсов для выпадающего списка
     */
    public static function getCoursesList()
    {
        return self::find()
            ->select(['name', 'id'])
            ->where(['is_active' => true])
            ->indexBy('id')
            ->column();
    }

    /**
     * Форматированная цена
     */
    public function getFormattedPrice()
    {
        return Yii::$app->formatter->asCurrency($this->price);
    }

    /**
     * Форматированная продолжительность
     */
    public function getFormattedDuration()
    {
        $weeks = ceil($this->duration_hours / 40); // 40 часов в неделю
        return $this->duration_hours . ' часов (' . $weeks . ' недель)';
    }
}