<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\behaviors\TimestampBehavior;
use yii\db\Expression;

/**
 * Application model
 *
 * @property integer $id
 * @property integer $user_id
 * @property integer $course_id
 * @property string $course_name
 * @property string $desired_start_date
 * @property string $payment_method
 * @property string $status
 * @property string $feedback
 * @property integer $rating
 * @property string $created_at
 * @property string $updated_at
 *
 * @property User $user
 * @property Course $course
 */
class Application extends ActiveRecord
{
    const STATUS_NEW = 'new';
    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_COMPLETED = 'completed';

    const PAYMENT_CASH = 'cash';
    const PAYMENT_TRANSFER = 'transfer';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'applications';
    }

    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            [
                'class' => TimestampBehavior::class,
                'createdAtAttribute' => 'created_at',
                'updatedAtAttribute' => 'updated_at',
                'value' => new Expression('NOW()'),
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['course_id', 'desired_start_date', 'payment_method'], 'required'],
            [['user_id', 'course_id', 'rating'], 'integer'],
            [['desired_start_date'], 'date', 'format' => 'php:Y-m-d'],
            ['desired_start_date', 'validateStartDate'],
            [['course_name'], 'string', 'max' => 200],
            [['payment_method'], 'in', 'range' => [self::PAYMENT_CASH, self::PAYMENT_TRANSFER]],
            [['status'], 'in', 'range' => [self::STATUS_NEW, self::STATUS_IN_PROGRESS, self::STATUS_COMPLETED]],
            [['feedback'], 'string'],
            ['rating', 'integer', 'min' => 1, 'max' => 5],
            [['user_id'], 'exist', 'skipOnError' => true, 'targetClass' => User::class, 'targetAttribute' => ['user_id' => 'id']],
            [['course_id'], 'exist', 'skipOnError' => true, 'targetClass' => Course::class, 'targetAttribute' => ['course_id' => 'id']],
        ];
    }

    /**
     * Валидация даты начала обучения
     */
    public function validateStartDate($attribute, $params)
    {
        if (!$this->hasErrors()) {
            $startDate = strtotime($this->desired_start_date);
            $today = strtotime(date('Y-m-d'));
            
            // Проверяем, что дата не в прошлом
            if ($startDate < $today) {
                $this->addError($attribute, 'Дата начала обучения не может быть в прошлом.');
            }
            
            // Проверяем, что дата не слишком далеко в будущем (максимум 1 год)
            $maxDate = strtotime('+1 year', $today);
            if ($startDate > $maxDate) {
                $this->addError($attribute, 'Дата начала обучения не может быть позже чем через год.');
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID заявки',
            'user_id' => 'Пользователь',
            'course_id' => 'Курс',
            'course_name' => 'Название курса',
            'desired_start_date' => 'Желаемая дата начала',
            'payment_method' => 'Способ оплаты',
            'status' => 'Статус',
            'feedback' => 'Отзыв',
            'rating' => 'Оценка (1-5)',
            'created_at' => 'Дата подачи',
            'updated_at' => 'Дата обновления',
            'userFullName' => 'ФИО',
            'coursePrice' => 'Стоимость курса',
        ];
    }

    /**
     * Получить пользователя
     */
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    /**
     * Получить курс
     */
    public function getCourse()
    {
        return $this->hasOne(Course::class, ['id' => 'course_id']);
    }

    /**
     * Получить ФИО пользователя
     */
    public function getUserFullName()
    {
        return $this->user ? $this->user->full_name : 'Неизвестно';
    }

    /**
     * Получить стоимость курса
     */
    public function getCoursePrice()
    {
        return $this->course ? $this->course->price : 0;
    }

    /**
     * Получить список статусов
     */
    public static function getStatusList()
    {
        return [
            self::STATUS_NEW => 'Новая',
            self::STATUS_IN_PROGRESS => 'Идет обучение',
            self::STATUS_COMPLETED => 'Обучение завершено',
        ];
    }

    /**
     * Получить список способов оплаты
     */
    public static function getPaymentMethods()
    {
        return [
            self::PAYMENT_CASH => 'Наличными',
            self::PAYMENT_TRANSFER => 'Переводом по номеру телефона',
        ];
    }

    /**
     * Получить текст статуса
     */
    public function getStatusText()
    {
        $statuses = self::getStatusList();
        return $statuses[$this->status] ?? $this->status;
    }

    /**
     * Получить текст способа оплаты
     */
    public function getPaymentMethodText()
    {
        $methods = self::getPaymentMethods();
        return $methods[$this->payment_method] ?? $this->payment_method;
    }

    /**
     * Получить CSS класс для статуса
     */
    public function getStatusClass()
    {
        $classes = [
            self::STATUS_NEW => 'badge bg-primary',
            self::STATUS_IN_PROGRESS => 'badge bg-warning',
            self::STATUS_COMPLETED => 'badge bg-success',
        ];
        return $classes[$this->status] ?? 'badge bg-secondary';
    }

    /**
     * Проверить, можно ли оставить отзыв
     */
    public function canLeaveFeedback()
    {
        return $this->status === self::STATUS_COMPLETED && empty($this->feedback);
    }

    /**
     * Перед сохранением
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                // При создании новой заявки
                $this->user_id = Yii::$app->user->id;
                $this->status = self::STATUS_NEW;
                
                // Получаем название курса
                $course = Course::findOne($this->course_id);
                if ($course) {
                    $this->course_name = $course->name;
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Получить заявки пользователя
     */
    public static function getUserApplications($userId)
    {
        return self::find()
            ->where(['user_id' => $userId])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();
    }

    /**
     * Получить все заявки (для администратора)
     */
    public static function getAllApplications()
    {
        return self::find()
            ->with(['user', 'course'])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();
    }
}