<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\AccessControl;
use app\models\Application;
use app\models\Course;

class ApplicationController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
        ];
    }

    /**
     * Список заявок пользователя
     */
    public function actionIndex()
    {
        $applications = Application::find()
            ->where(['user_id' => Yii::$app->user->id])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();

        return $this->render('index', [
            'applications' => $applications,
        ]);
    }

    /**
     * Создание новой заявки
     */
    public function actionCreate()
    {
        $model = new Application();
        $courses = Course::find()->where(['is_active' => 1])->all();

        if ($model->load(Yii::$app->request->post())) {
            $model->user_id = Yii::$app->user->id;
            $model->status = 'new';
            
            // Получаем название курса
            $course = Course::findOne($model->course_id);
            if ($course) {
                $model->course_name = $course->name;
            }
            
            if ($model->save()) {
                Yii::$app->session->setFlash('success', 
                    'Заявка успешно создана и отправлена на рассмотрение. ' .
                    'Номер вашей заявки: ' . $model->id
                );
                return $this->redirect(['index']);
            } else {
                Yii::$app->session->setFlash('error', 'Ошибка при создании заявки.');
            }
        }

        return $this->render('create', [
            'model' => $model,
            'courses' => $courses,
        ]);
    }

    /**
     * Просмотр заявки
     */
    public function actionView($id)
    {
        $model = $this->findModel($id);
        
        // Проверяем, что заявка принадлежит текущему пользователю
        if ($model->user_id != Yii::$app->user->id) {
            throw new NotFoundHttpException('Заявка не найдена.');
        }

        return $this->render('view', [
            'model' => $model,
        ]);
    }

    /**
     * Оставить отзыв
     */
    public function actionFeedback($id)
    {
        $model = $this->findModel($id);
        
        // Проверяем, что заявка принадлежит текущему пользователю
        if ($model->user_id != Yii::$app->user->id) {
            throw new NotFoundHttpException('Заявка не найдена.');
        }
        
        // Проверяем, можно ли оставить отзыв
        if ($model->status != 'completed' || !empty($model->feedback)) {
            Yii::$app->session->setFlash('error', 
                'Отзыв можно оставить только для завершенных курсов.'
            );
            return $this->redirect(['index']);
        }

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Спасибо! Ваш отзыв успешно сохранен.');
            return $this->redirect(['index']);
        }

        return $this->render('feedback', [
            'model' => $model,
        ]);
    }

    /**
     * Поиск заявки
     */
    protected function findModel($id)
    {
        if (($model = Application::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('Заявка не найдена.');
    }
}